﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Npgsql;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    

    [Authorize]
    [Route("api/driver-Detail")]
    [Consumes("application/json")]
    [Produces("application/json")]

    public class DriverDetailController:BaseController
    {

        private readonly IDriverDetailService driverDetailService;
        private readonly IAuditLogService auditLogServices;


        public DriverDetailController(IDriverDetailService driverDetailServices, IAuditLogService auditLogService)
        {
            this.driverDetailService = driverDetailServices;
            this.auditLogServices = auditLogService;
        }


        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> Add([FromBody] DriverDetailModel model)
        {
            model = (DriverDetailModel)EmptyFilter.Handler(model);
            
            var response = await driverDetailService.InsertAsync(model);
            if (response > 0)
            {

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DriverDetail,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.CreatedByName} has been added Driver Detail  {model.DriverDetailName}.",
                    
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }


            return this.Success(response);
        }
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<DriverDetailModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] DriverDetailModel model, [FromHeader] LocationHeader location)
        {
            
            var ambulance = await this.driverDetailService.FetchAllAsync(model);
            return ambulance == null ? this.ServerError() : this.Success(ambulance);
        }

        [HttpPost]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] DriverDetailModel model, [FromHeader] LocationHeader header)
        {
            model = (DriverDetailModel)EmptyFilter.Handler(model);
            var response = await this.driverDetailService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DriverDetail,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.CreatedByName} has been updated Driver Detail   {model.DriverDetailName}.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            return this.Success(response);
        }



        [HttpPut]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] DriverDetailModel model, [FromHeader] LocationHeader header)
        {

            var response = await this.driverDetailService.ModifyStatusAsync(model.DriverDetailId, Convert.ToInt32(model.ModifiedBy), model.Active);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.DriverDetail,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.Now,
                LogDescription = $"{model.ModifiedByName} has been updated the  status  to Driver Detail   {model.DriverDetailName} {(model.Active ? "'Active'" : "'Inactive'")}.",
                LocationId = Convert.ToInt32(header.LocationId) 
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);
        }


        //[HttpPost]
        //[Route("delete")]
        //[ProducesResponseType(typeof(string), 200)]
        //[ProducesResponseType(409)]
        //[ProducesResponseType(500)]
        //public async Task<ActionResult> DeleteAsync([FromBody] DriverDetailModel model, [FromHeader] LocationHeader header)
        //{
        //    try
        //    {
        //        var response = await this.driverDetailService.DeleteAsync(model.DriverDetailId);
        //        if (response == 0)
        //        {
        //            return this.ServerError();
        //        }

        //        var auditLogModel = new AuditLogModel
        //        {
        //            AccountId = model.ModifiedBy,
        //            LogTypeId = (int)LogTypes.DriverDetail,
        //            LogFrom = (int)AccountType.Administrator,
        //            LogDate = DateTime.Now,
        //            LogDescription = $"{model.ModifiedByName} has been deleted driver no {model.DriverDetailName}.",
        //            LocationId = Convert.ToInt32(header.LocationId)
        //        };
        //        await this.auditLogServices.LogAsync(auditLogModel);

        //        return this.Success(response);
        //    }
        //    catch (NpgsqlException exception)
        //    {
        //        if (exception.Message.Contains("violates foreign key constraint"))
        //        {
        //            return this.Conflict("ambulance can't be deleted. Please contact Administrator.");
        //        }

        //        return this.ServerError();
        //    }
        //}

    }

}
